<?php

/**
 * Convertisseur de contenu d'une base sqlite d'article Blogotext 2
 * vers des fichiers XML compatible Pluxml 5
 *
 * Nécessite :
 *  - fichier "database.sqlite" dans le même dossier que ce script
 *  - sous-dossier "articles" VIDE
 *  - sous-dossier "commentaires" VIDE
 *  - PHP et pdo_sqlite (comme Blogotext)
 *
 * Valeurs forcées par défaut pour tous les articles :
 *  - user ID = 001
 *  - catégorie ID = 000
 *  - allow_com = 1
 *  - les articles non publiés dans Blogotext seront à dépublier manuellement dans Pluxml
 *  
 * Valeurs forcées par défaut pour tous les commentaires :
 *  - type = normal
 *
 * ATTENTION : certains éléments ne sont pas pris en compte par ce script :
 *  - ce script ne vide pas les dossiers "articles" et "commentaires"
 *  - les "links" de Blogotext ne sont pas repris
 *  - les liens internes au blogs vers d'autres articles/commentaires ne sont pas corrigés
 *  - le formattage "wiki" de Blogotext n'est pas convertit en HTML
 *  - évidemment, les images et autres fichiers ne sont pas repris ici
 */

?><!doctype html>
<html lang="fr-FR">
<head>
    <meta charset="UTF-8">
    <title>Transfert de Blogotext 2 vers pluxml 5</title>
</head>
<body>
<p>Etapes :</p>
<ul>
<?php

/**
 * Fonctions du convertisseur
 */
function show($string) {
    echo "\t<li>".$string."</li>\r\n";
}
function formatArticle($article) {
    $text = "<?xml version='1.0' encoding='UTF-8'?>"
        ."\r\n<document>"
        ."\r\n\t<title><![CDATA[".$article['bt_title']."]]></title>"
        ."\r\n\t<allow_com>1</allow_com>"
        ."\r\n\t<template><![CDATA[article.php]]></template>"
        ."\r\n\t<chapo><![CDATA[".trim($article['bt_abstract'])."]]></chapo>"
        ."\r\n\t<content><![CDATA[".trim($article['bt_content'])."]]></content>"
        ."\r\n\t<tags><![CDATA[".trim($article['bt_categories'])."]]></tags>"
        ."\r\n\t<meta_description><![CDATA[]]></meta_description>"
        ."\r\n\t<meta_keywords><![CDATA[]]></meta_keywords>"
        ."\r\n\t<title_htmltag><![CDATA[]]></title_htmltag>"
        ."\r\n</document>";

    return $text;
}
function formatComment($comment) {
    $text = "<?xml version='1.0' encoding='UTF-8'?>"
        ."\r\n<comment>"
        ."\r\n\t<author><![CDATA[".$comment['bt_author']."]]></author>"
        ."\r\n\t<type>normal</type>"
        ."\r\n\t<ip></ip>"
        ."\r\n\t<mail><![CDATA[".trim($comment['bt_email'])."]]></mail>"
        ."\r\n\t<site><![CDATA[".trim($comment['bt_webpage'])."]]></site>"
        ."\r\n\t<content><![CDATA[".trim($comment['bt_content'])."]]></content>"
        ."\r\n</comment>";

    return $text;
}

/**
 * Fonctions reprises de Pluxml
 */
function removeAccents($str,$charset='utf-8') {

    $str = htmlentities($str, ENT_NOQUOTES, $charset);
    $str = preg_replace('#\&([A-za-z])(?:acute|cedil|circ|grave|ring|tilde|uml|uro)\;#', '\1', $str);
    $str = preg_replace('#\&([A-za-z]{2})(?:lig)\;#', '\1', $str); # pour les ligatures e.g. '&oelig;'
    $str = preg_replace('#\&[^;]+\;#', '', $str); # supprime les autres caractères
    return $str;
}
function title2filename($str) {

    $str = strtolower(removeAccents($str,'utf-8'));
    $str = str_replace('|','',$str);
    $str = preg_replace('/\.{2,}/', '.', $str);
    $str = preg_replace('/[^[:alnum:]|.|_]+/',' ',$str);
    return strtr(ltrim(trim($str),'.'), ' ', '-');
}

/**
 * Connexion à la base de Blogotext
 */
$BtDb = 'database.sqlite';
show('Connexion à la base "'.$BtDb.'"...');
try {
    $db = new PDO('sqlite:'.$BtDb);
} catch(PDOException $e) {
    show($e->getMessage());
}

/**
 * Gestion des articles
 */
$result = $db->query("SELECT * FROM articles ORDER BY bt_date ASC");
$articles = $result->fetchAll(PDO::FETCH_ASSOC);
show(count($articles).' articles à traiter...');

$articleDir = 'articles';
if(!is_dir($articleDir)) mkdir($articleDir);
$pluxmlId = 1;
$idCorrespondance = array();
foreach($articles as $key => $article) {
    $idCorrespondance[$article['bt_id']] = $pluxmlId;
    $url = title2filename($article['bt_title']);
    $fileName = $articleDir.'/'.str_pad($pluxmlId, 4, "0", STR_PAD_LEFT).'.000.001.'.substr($article['bt_date'], 0, -2).'.'.$url.'.xml';
    file_put_contents($fileName, formatArticle($article));
    $pluxmlId++;
}

/**
 * Gestion des commentaires
 */
$result = $db->query("SELECT * FROM commentaires ORDER BY bt_article_id ASC,  bt_id ASC");
$comments = $result->fetchAll(PDO::FETCH_ASSOC);
show(count($comments).' commentaires à traiter...');

$commentDir = 'commentaires';
if(!is_dir($commentDir)) mkdir($commentDir);
$pluxmlId = 1;
$pluxmlArticleId = -1;
foreach($comments as $key => $comment) {
    //reset comment id to 1 for every article
    if($idCorrespondance[$comment['bt_article_id']] != $pluxmlArticleId) {
        $pluxmlArticleId = $idCorrespondance[$comment['bt_article_id']];
        $pluxmlId = 1;
    }

    $d = array(
        'y' => substr($comment['bt_id'], 0, 4),
        'm' => substr($comment['bt_id'], 4, 2),
        'd' => substr($comment['bt_id'], 6, 2),
        'h' => substr($comment['bt_id'], 8, 2),
        'i' => substr($comment['bt_id'], 10, 2),
        's' => substr($comment['bt_id'], 12, 2)
    );
    $fileName = $commentDir.'/'.str_pad($idCorrespondance[$comment['bt_article_id']], 4, "0", STR_PAD_LEFT)
        .'.'.mktime($d['h'], $d['i'], $d['s'], $d['m'], $d['d'], $d['y'])
        .'-'.$pluxmlId.'.xml';
    file_put_contents($fileName, formatComment($comment));
    $pluxmlId++;
}


/**
 * Handle links
 */
$result = $db->query("SELECT * FROM links ORDER BY bt_id ASC");
$links = $result->fetchAll(PDO::FETCH_ASSOC);
show(count($links).' liens à traiter...');
if(count($links) > 0) {
    show('ATTENTION : les liens ne sont pas traités par ce script !');
}

?>
</ul>
</body>
</html>